package tocraft.walkers.api.data.variants;

import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.NotNull;
import tocraft.walkers.Walkers;
import tocraft.walkers.api.variant.TypeProvider;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_5250;
import net.minecraft.class_7923;

// this is amazing
public class NBTTypeProvider<T extends class_1309> extends TypeProvider<T> {
    public static final Codec<NBTTypeProvider<?>> CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            Codec.INT.optionalFieldOf("fallback", 0).forGetter(NBTTypeProvider::getFallbackData),
            Codec.INT.optionalFieldOf("range", -1).forGetter(NBTTypeProvider::getRange),
            Codec.either(Codec.list(NBTEntry.CODEC), AdvancedNBTEntries.CODEC).fieldOf("nbt").forGetter(o -> o.nbtEntryList),
            Codec.unboundedMap(Codec.STRING, Codec.STRING).optionalFieldOf("names", new HashMap<>()).forGetter(o -> o.nameMap)
    ).apply(instance, instance.stable(NBTTypeProvider::new)));

    private final int fallback;
    private final int range;
    private final Either<List<NBTEntry<?>>, AdvancedNBTEntries> nbtEntryList;
    private final Map<String, String> nameMap;

    public NBTTypeProvider(int fallback, int range, Either<List<NBTEntry<?>>, AdvancedNBTEntries> nbtEntryList, Map<String, String> nameMap) {
        this.fallback = fallback;
        this.nbtEntryList = nbtEntryList;
        this.nameMap = nameMap;
        if (range >= 0 && fallback <= range) {
            this.range = range;
        } else if (nbtEntryList.left().isPresent()) {
            switch (nbtEntryList.left().get().get(0).nbtType().toUpperCase()) {
                case "BOOL", "BOOLEAN" -> this.range = 1;
                default -> this.range = fallback;
            }
        } else if (nbtEntryList.right().isPresent()) {
            this.range = nbtEntryList.right().get().highestId();
        } else {
            this.range = 0;
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public int getVariantData(T entity) {
        class_2487 tag = new class_2487();
        entity.method_5662(tag);
        List<List<Integer>> validValues = new ArrayList<>();
        if (nbtEntryList.left().isPresent()) {
            for (NBTEntry<?> nbtEntry : nbtEntryList.left().get()) {
                if (tag.method_10545(nbtEntry.nbtField())) {
                    switch (nbtEntry.nbtType().toUpperCase()) {
                        case "BOOL", "BOOLEAN" ->
                                validValues.add(((NBTEntry<Boolean>) nbtEntry).getIndex(tag.method_10577(nbtEntry.nbtField())));
                        case "STRING" ->
                                validValues.add(((NBTEntry<String>) nbtEntry).getIndex(tag.method_10558(nbtEntry.nbtField())));
                        case "INT", "INTEGER" ->
                                validValues.add(((NBTEntry<Integer>) nbtEntry).getIndex(tag.method_10550(nbtEntry.nbtField())));
                    }
                }
            }
        }

        // support AdvancedNBTEntries
        else if (nbtEntryList.right().isPresent()) {
            validValues.add(List.of(nbtEntryList.right().get().getData(tag)));
        }

        // check if data applies to all nbt fields
        List<Integer> validData = getValidDataValues(validValues);
        if (!validData.isEmpty()) {
            if (validData.size() > 1) {
                Walkers.LOGGER.error("{}: found too much valid variant ids: {} for entity: {}", getClass().getSimpleName(), validData.toArray(Integer[]::new), entity.method_5864().method_5882());
            }
            return validData.get(0);
        } else {
            Walkers.LOGGER.error("{}: No Variant for entity type {} found.", getClass().getSimpleName(), entity.method_5864().method_5882());
            return getFallbackData();
        }
    }

    @NotNull
    private static List<Integer> getValidDataValues(List<List<Integer>> validValues) {
        List<Integer> validData = new ArrayList<>();
        for (List<Integer> validValue : validValues) {
            for (Integer i : validValue) {
                boolean invalid = false;
                for (List<Integer> value : validValues) {
                    if (!value.contains(i)) {
                        invalid = true;
                        break;
                    }
                }
                if (!invalid) {
                    if (!validData.contains(i)) {
                        validData.add(i);
                    }
                }
            }
        }
        return validData;
    }

    @SuppressWarnings("unchecked")
    @Override
    public T create(class_1299<T> type, class_1937 world, int data) {
        class_2487 tag = new class_2487();

        if (nbtEntryList.left().isPresent()) {
            for (NBTEntry<?> nbtEntry : nbtEntryList.left().get()) {
                Object value = nbtEntry.getValue(data);
                if (value instanceof Integer intValue) {
                    tag.method_10569(nbtEntry.nbtField(), intValue);
                } else if (value instanceof String stringValue) {
                    tag.method_10582(nbtEntry.nbtField(), stringValue);
                } else if (value instanceof Boolean booleanValue) {
                    tag.method_10556(nbtEntry.nbtField(), booleanValue);
                } else if (value == null) {
                    Walkers.LOGGER.error("{}: variant parameter for {} not found.", getClass().getSimpleName(), type.method_5882());
                }
            }
        }
        // support AdvancedNBTEntries
        else if (nbtEntryList.right().isPresent()) {
            nbtEntryList.right().get().fromData(tag, data);
        }

        class_2487 compoundTag = tag.method_10553();
        compoundTag.method_10582("id", class_7923.field_41177.method_10221(type).toString());
        return (T) class_1299.method_17842(compoundTag, world, entity -> entity);
    }

    @Override
    public int getFallbackData() {
        return fallback;
    }

    @Override
    public int getRange() {
        return range;
    }

    @Override
    public class_2561 modifyText(T entity, class_5250 text) {
        if (nameMap.containsKey(String.valueOf(getVariantData(entity))))
            return class_2561.method_43469(nameMap.get(String.valueOf(getVariantData(entity))), text);
        else
            return text;
    }
}
