package tocraft.walkers.api.data.variants;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.NotNull;
import tocraft.walkers.Walkers;
import tocraft.walkers.api.variant.TypeProvider;

import java.util.*;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_1937;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_5250;
import net.minecraft.class_7923;

// this is amazing
public class NBTTypeProvider<T extends class_1309> extends TypeProvider<T> {
    public static Codec<NBTTypeProvider<?>> CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            Codec.INT.optionalFieldOf("fallback", 0).forGetter(NBTTypeProvider::getFallbackData),
            Codec.INT.optionalFieldOf("range").forGetter(o -> Optional.of(o.getRange())),
            Codec.list(NBTEntry.CODEC).fieldOf("nbt").forGetter(o -> o.nbtEntryList),
            Codec.unboundedMap(Codec.STRING, Codec.STRING).optionalFieldOf("names", new HashMap<>()).forGetter(o -> o.nameMap)
    ).apply(instance, instance.stable(NBTTypeProvider::new)));

    private final int fallback;
    private final int range;
    private final List<NBTEntry<?>> nbtEntryList;
    private final Map<String, String> nameMap;

    NBTTypeProvider(int fallback, Optional<Integer> range, List<NBTEntry<?>> nbtEntryList, Map<String, String> nameMap) {
        this(fallback, range.orElseGet(() -> {
            switch (nbtEntryList.get(0).nbtType.toUpperCase()) {
                case "BOOL", "BOOLEAN" -> {
                    return 1;
                }
                default -> {
                    return fallback;
                }
            }
        }), nbtEntryList, nameMap);
    }

    NBTTypeProvider(int fallback, int range, List<NBTEntry<?>> nbtEntryList, Map<String, String> nameMap) {
        this.fallback = fallback;
        this.range = range;
        this.nbtEntryList = nbtEntryList;
        this.nameMap = nameMap;
    }

    @SuppressWarnings("unchecked")
    @Override
    public int getVariantData(T entity) {
        class_2487 tag = new class_2487();
        entity.method_5662(tag);
        List<List<Integer>> validValues = new ArrayList<>();
        for (NBTEntry<?> nbtEntry : nbtEntryList) {
            if (tag.method_10545(nbtEntry.nbtField())) {
                switch (nbtEntry.nbtType().toUpperCase()) {
                    case "BOOL", "BOOLEAN" ->
                            validValues.add(((NBTEntry<Boolean>) nbtEntry).getIndex(tag.method_10577(nbtEntry.nbtField())));
                    case "STRING" ->
                            validValues.add(((NBTEntry<String>) nbtEntry).getIndex(tag.method_10558(nbtEntry.nbtField())));
                    case "INT", "INTEGER" ->
                            validValues.add(((NBTEntry<Integer>) nbtEntry).getIndex(tag.method_10550(nbtEntry.nbtField())));
                }
            }
        }

        // check if data applies to all nbt fields
        List<Integer> validData = getValidDataValues(validValues);
        if (!validData.isEmpty()) {
            if (validData.size() > 1) {
                Walkers.LOGGER.error("{}: found too much valid variant ids: {} for entity: {}", getClass().getSimpleName(), validData.size(), entity.method_5864().method_5882());
            }
            return validData.get(0);
        }
        Walkers.LOGGER.error("{}: parameter for the Variant not found.", getClass().getSimpleName());
        return getFallbackData();
    }

    @NotNull
    private static List<Integer> getValidDataValues(List<List<Integer>> validValues) {
        List<Integer> validData = new ArrayList<>();
        for (List<Integer> validValue : validValues) {
            for (Integer i : validValue) {
                boolean invalid = false;
                for (List<Integer> value : validValues) {
                    if (!value.contains(i)) {
                        invalid = true;
                        break;
                    }
                }
                if (!invalid) {
                    validData.add(i);
                }
            }
        }
        return validData;
    }

    @SuppressWarnings("unchecked")
    @Override
    public T create(class_1299<T> type, class_1937 world, int data) {
        class_2487 tag = new class_2487();

        for (NBTEntry<?> nbtEntry : nbtEntryList) {
            Object value = nbtEntry.getValue(data);
            if (value instanceof Integer intValue) {
                tag.method_10569(nbtEntry.nbtField(), intValue);
            } else if (value instanceof String stringValue) {
                tag.method_10582(nbtEntry.nbtField(), stringValue);
            } else if (value instanceof Boolean booleanValue) {
                tag.method_10556(nbtEntry.nbtField(), booleanValue);
            }
        }

        class_2487 compoundTag = tag.method_10553();
        compoundTag.method_10582("id", class_7923.field_41177.method_10221(type).toString());
        return (T) class_1299.method_17842(compoundTag, world, entity -> entity);
    }

    @Override
    public int getFallbackData() {
        return fallback;
    }

    @Override
    public int getRange() {
        return range;
    }

    @Override
    public class_2561 modifyText(T entity, class_5250 text) {
        if (nameMap.containsKey(String.valueOf(getVariantData(entity))))
            return class_2561.method_43469(nameMap.get(String.valueOf(getVariantData(entity))), text);
        else
            return text;
    }

    @SuppressWarnings("unchecked")
    public record NBTEntry<T>(String nbtType, String nbtField, Map<Integer, T> parameterList, boolean isMutable) {
        public static Codec<NBTEntry<?>> CODEC = RecordCodecBuilder.create((instance) -> instance.group(
                Codec.STRING.fieldOf("nbt_type").forGetter(NBTEntry::nbtType),
                Codec.STRING.fieldOf("nbt_field").forGetter(NBTEntry::nbtField),
                Codec.unboundedMap(Codec.STRING, Codec.STRING).optionalFieldOf("parameters", new HashMap<>()).forGetter(o -> new HashMap<>()),
                Codec.BOOL.optionalFieldOf("is_mutable", false).forGetter(NBTEntry::isMutable)
        ).apply(instance, instance.stable((nbtType, nbtField, parameters, isMutable) -> {
            switch (nbtType.toUpperCase()) {
                case "INT", "INTEGER" -> {
                    return new NBTEntry<>(nbtType, nbtField, new HashMap<>() {
                        {
                            parameters.forEach((key, value) -> put(Integer.valueOf(key), Integer.valueOf(value)));
                        }
                    }, isMutable);
                }
                case "BOOL", "BOOLEAN" -> {
                    return new NBTEntry<>(nbtType, nbtField, new HashMap<>() {
                        {
                            parameters.forEach((key, value) -> put(Integer.valueOf(key), Boolean.valueOf(value)));
                        }
                    }, isMutable);
                }
                default -> {
                    return new NBTEntry<>(nbtType, nbtField, new HashMap<>() {
                        {
                            parameters.forEach((key, value) -> put(Integer.valueOf(key), value));
                        }
                    }, isMutable);
                }
            }
        })));

        public T getValue(int index) {
            if (parameterList.containsKey(index)) {
                return parameterList.get(index);
            }
            switch (nbtType.toUpperCase()) {
                case "INT", "INTEGER" -> {
                    return (T) (Object) index;
                }
                case "BOOL", "BOOLEAN" -> {
                    // check if index is odd
                    if (index == 1) return (T) (Object) true;
                    else return (T) (Object) false;
                }
            }
            Walkers.LOGGER.error("{}: variant parameter not found.", getClass().getSimpleName());
            return null;
        }

        public List<Integer> getIndex(T value) {
            List<Integer> index = new ArrayList<>();
            if (!parameterList.isEmpty()) {
                if (isMutable && value instanceof String) {
                    class_5250 tagDataMutable = class_2561.class_2562.method_10873((String) value);
                    if (tagDataMutable != null) {
                        value = (T) tagDataMutable.getString();
                    }
                }
                for (int i : parameterList.keySet()) {
                    T parameterT = parameterList.get(i);
                    if (isMutable && parameterT instanceof String) {
                        class_5250 parameterMutable = class_2561.class_2562.method_10873((String) parameterT);
                        if (parameterMutable != null) {
                            parameterT = (T) parameterMutable.getString();
                        }
                    }
                    if (value.equals(parameterT)) {
                        index.add(i);
                    }
                }
            }
            if (index.isEmpty()) {
                switch (nbtType.toUpperCase()) {
                    case "BOOL", "BOOLEAN" -> {
                        if ((Boolean) value) {
                            index.add(1);
                        } else index.add(0);
                    }
                    case "INT", "INTEGER" -> index.add((Integer) value);
                }
            }
            return index;
        }
    }
}
