package tocraft.walkers.api.data.traits;

import com.google.gson.*;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import tocraft.craftedcore.data.SynchronizedJsonReloadListener;
import tocraft.craftedcore.platform.PlatformData;
import tocraft.walkers.Walkers;
import tocraft.walkers.traits.ShapeTrait;
import tocraft.walkers.traits.TraitRegistry;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_6862;

public class TraitDataManager extends SynchronizedJsonReloadListener {
    public static final Gson GSON = new GsonBuilder().registerTypeAdapter(class_2960.class, new class_2960.class_2961()).create();
    private final boolean isDeprecatedSkills;

    public TraitDataManager() {
        this(false);
    }

    public TraitDataManager(boolean isDeprecatedSkills) {
        super(GSON, Walkers.MODID + (isDeprecatedSkills ? "/skills" : "/traits"));
        this.isDeprecatedSkills = isDeprecatedSkills;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void onApply(Map<class_2960, JsonElement> map) {
        // prevent duplicates and the registration of removed entries
        TraitRegistry.clearAll();
        TraitRegistry.registerDefault();

        for (Map.Entry<class_2960, JsonElement> mapEntry : map.entrySet()) {
            TraitList traitList = traitListFromJson(mapEntry.getValue().getAsJsonObject());

            if (!traitList.isEmpty()) {
                if (isDeprecatedSkills) {
                    Walkers.LOGGER.error("{}: Using the skills directory & key is deprecated. Please merge to 'trait's.", getClass().getSimpleName());
                }

                if (traitList.requiredMod() == null || traitList.requiredMod().isBlank() || PlatformData.isModLoaded(traitList.requiredMod())) {
                    // entity types
                    for (class_1299<class_1309> entityType : traitList.entityTypes()) {
                        TraitRegistry.registerByType(entityType, traitList.traitList().stream().map(trait -> (ShapeTrait<class_1309>) trait).toList());
                    }

                    if (!traitList.entityTypes().isEmpty())
                        Walkers.LOGGER.info("{}: {} registered for {}", getClass().getSimpleName(), traitList.entityTypes(), traitList.traitList().stream().map(trait -> trait.getClass().getSimpleName()).toArray(String[]::new));

                    // entity tags
                    for (class_6862<class_1299<?>> entityTag : traitList.entityTags()) {
                        TraitRegistry.registerByTag(entityTag, traitList.traitList().stream().map(trait -> (ShapeTrait<class_1309>) trait).toList());
                    }

                    if (!traitList.entityTags().isEmpty())
                        Walkers.LOGGER.info("{}: {} registered for {}", getClass().getSimpleName(), traitList.entityTags(), traitList.traitList().stream().map(trait -> trait.getClass().getSimpleName()).toArray(String[]::new));
                }
            }
        }
    }

    public static Codec<TraitList> TRAIT_LIST_CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            Codec.STRING.optionalFieldOf("required_mod", "").forGetter(TraitList::requiredMod),
            Codec.list(class_2960.field_25139).optionalFieldOf("entity_types", new ArrayList<>()).forGetter(TraitList::entityTypeKeys),
            Codec.list(class_2960.field_25139).optionalFieldOf("entity_tags", new ArrayList<>()).forGetter(TraitList::entityTagKeys),
            Codec.list(TraitRegistry.getTraitCodec()).fieldOf("traits").forGetter(TraitList::traitList)
    ).apply(instance, instance.stable(TraitList::new)));

    public Codec<TraitList> SKILL_LIST_CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            Codec.STRING.optionalFieldOf("required_mod", "").forGetter(TraitList::requiredMod),
            Codec.list(class_2960.field_25139).optionalFieldOf("entity_types", new ArrayList<>()).forGetter(TraitList::entityTypeKeys),
            Codec.list(class_2960.field_25139).optionalFieldOf("entity_tags", new ArrayList<>()).forGetter(TraitList::entityTagKeys),
            Codec.list(TraitRegistry.getTraitCodec()).fieldOf("skills").forGetter(TraitList::traitList)
    ).apply(instance, instance.stable(TraitList::new)));

    protected TraitList traitListFromJson(JsonObject json) {
        return (isDeprecatedSkills ? SKILL_LIST_CODEC : TRAIT_LIST_CODEC).parse(JsonOps.INSTANCE, json).getOrThrow(false, msg -> {
            throw new JsonParseException(msg);
        });
    }

    @SuppressWarnings("unused")
    public record TraitList(String requiredMod, List<class_2960> entityTypeKeys,
                            List<class_2960> entityTagKeys,
                            List<ShapeTrait<?>> traitList) {

        public TraitList(List<class_1299<?>> entityTypeKeys, List<class_6862<class_1299<?>>> entityTagKeys, List<ShapeTrait<?>> traitList, String requiredMod) {
            this(requiredMod, entityTypeKeys.stream().map(class_1299::method_5890).toList(), entityTagKeys.stream().map(class_6862::comp_327).toList(), traitList);
        }

        @SuppressWarnings("unchecked")
        public List<class_1299<class_1309>> entityTypes() {
            if (entityTagKeys.contains(new class_2960("alexsmobs:bald_eagle")))
                Walkers.LOGGER.warn("got that far");
            return entityTypeKeys.stream().filter(class_2378.field_11145::method_10250).map(type -> (class_1299<class_1309>) class_2378.field_11145.method_10223(type)).toList();
        }

        public List<class_6862<class_1299<?>>> entityTags() {
            return entityTagKeys().stream().map(tag -> class_6862.method_40092(class_2378.field_25107, tag)).toList();
        }

        public boolean isEmpty() {
            return (entityTypeKeys().isEmpty() && entityTagKeys().isEmpty()) || traitList().isEmpty();
        }
    }
}
