package tocraft.remorphed.screen;

import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.components.Tooltip;
import net.minecraft.client.gui.narration.NarratableEntry;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.network.chat.Component;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.Mob;
import net.minecraft.world.entity.player.Player;
import tocraft.remorphed.Remorphed;
import tocraft.remorphed.impl.RemorphedPlayerDataProvider;
import tocraft.remorphed.mixin.accessor.ScreenAccessor;
import tocraft.remorphed.screen.widget.EntityWidget;
import tocraft.remorphed.screen.widget.PlayerWidget;
import tocraft.remorphed.screen.widget.SearchWidget;
import tocraft.remorphed.screen.widget.SpecialShapeWidget;
import tocraft.walkers.Walkers;
import tocraft.walkers.api.PlayerShape;
import tocraft.walkers.api.variant.ShapeType;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;

@Environment(EnvType.CLIENT)
public class RemorphedScreen extends Screen {
    private final List<ShapeType<?>> unlocked = new CopyOnWriteArrayList<>();
    private final Map<ShapeType<?>, Mob> renderEntities = new LinkedHashMap<>();
    private final List<EntityWidget<?>> entityWidgets = new CopyOnWriteArrayList<>();
    private final SearchWidget searchBar = createSearchBar();
    private final Button helpButton = createHelpButton();
    private final Button variantsButton = createVariantsButton();
    private final Button skillsButton = createSkillsButton();
    private final PlayerWidget playerButton = createPlayerButton();
    private final SpecialShapeWidget specialShapeButton = createSpecialShapeButton();
    private String lastSearchContents = "";

    public RemorphedScreen() {
        super(Component.m_237113_(""));
        super.m_6575_(Minecraft.m_91087_(), Minecraft.m_91087_().m_91268_().m_85445_(), Minecraft.m_91087_().m_91268_().m_85446_());
    }

    @Override
    public void m_7856_() {
        // don't initialize if the player is null
        if (f_96541_ == null) return;
        if (f_96541_.f_91074_ == null) {
            this.m_7379_();
            return;
        }

        m_142416_(searchBar);
        m_142416_(helpButton);
        m_142416_(variantsButton);
        m_142416_(skillsButton);
        m_142416_(playerButton);
        if (Walkers.hasSpecialShape(f_96541_.f_91074_.m_20148_()))
            m_142416_(specialShapeButton);

        new Thread(() -> {
            populateUnlockedRenderEntities(f_96541_.f_91074_);

            ShapeType<? extends LivingEntity> currentShape = ShapeType.from(PlayerShape.getCurrentShape(f_96541_.f_91074_));

            // handle favorites
            unlocked.sort((first, second) -> {
                if (Objects.equals(first, currentShape)) {
                        return -1;
                } else if (Objects.equals(second, currentShape)) {
                        return 1;
                    } else {
                        boolean firstIsFav = ((RemorphedPlayerDataProvider) f_96541_.f_91074_).remorphed$getFavorites().contains(first);
                        boolean secondIsFav = ((RemorphedPlayerDataProvider) f_96541_.f_91074_).remorphed$getFavorites().contains(second);
                        if (firstIsFav == secondIsFav)
                            return 0;
                        if (firstIsFav)
                            return -1;
                        else return 1;
                    }
                });

            // filter unlocked
            if (!Remorphed.displayVariantsInMenu) {
                List<ShapeType<?>> newUnlocked = new ArrayList<>();
                for (ShapeType<?> shapeType : unlocked) {
                    if (shapeType.equals(currentShape) || !newUnlocked.stream().map(ShapeType::getEntityType).toList().contains(shapeType.getEntityType())) {
                        newUnlocked.add(shapeType);
                    }
                }

                unlocked.clear();
                unlocked.addAll(newUnlocked);
            }

            populateEntityWidgets(unlocked);
        }, "cache entities").start();

        // implement search handler
        searchBar.m_94151_(text -> {
            m_94725_(searchBar);

            // Only re-filter if the text contents changed
            if (!lastSearchContents.equals(text)) {
                ((ScreenAccessor) this).getSelectables().removeIf(button -> button instanceof EntityWidget);
                m_6702_().removeIf(button -> button instanceof EntityWidget);

                List<ShapeType<?>> filtered = unlocked
                        .stream()
                        .filter(type -> text.isEmpty() || ShapeType.createTooltipText(renderEntities.get(type)).getString().toUpperCase().contains(text.toUpperCase()) || EntityType.m_20613_(type.getEntityType()).toString().toUpperCase().contains(text.toUpperCase()))
                        .collect(Collectors.toList());

                populateEntityWidgets(filtered);
            }

            lastSearchContents = text;
        });
    }

    @Override
    public void m_86412_(PoseStack context, int mouseX, int mouseY, float delta) {
        m_7333_(context);

        searchBar.m_86412_(context, mouseX, mouseY, delta);
        helpButton.m_86412_(context, mouseX, mouseY, delta);
        variantsButton.m_86412_(context, mouseX, mouseY, delta);
        skillsButton.m_86412_(context, mouseX, mouseY, delta);
        playerButton.m_86412_(context, mouseX, mouseY, delta);
        if (Walkers.hasSpecialShape(f_96541_.f_91074_.m_20148_()))
            specialShapeButton.m_86412_(context, mouseX, mouseY, delta);

        double scaledFactor = this.f_96541_.m_91268_().m_85449_();
        int top = 35;

        context.m_85836_();
        RenderSystem.m_69488_(
                (int) ((double) 0 * scaledFactor),
                (int) ((double) 0 * scaledFactor),
                (int) ((double) f_96543_ * scaledFactor),
                (int) ((double) (this.f_96544_ - top) * scaledFactor));

        for (EntityWidget<?> widget : entityWidgets) {
            if (widget.m_252907_() + widget.m_93694_() > top && widget.m_252907_() < getWindow().m_85446_()) {
                widget.m_86412_(context, mouseX, mouseY, delta);
            }
        }

        RenderSystem.m_69471_();

        context.m_85849_();
    }

    @Override
    public boolean m_6050_(double mouseX, double mouseY, double scrollY) {
        if (!entityWidgets.isEmpty()) {
            float firstPos = entityWidgets.get(0).m_252907_();
            EntityWidget<?> lastWidget = entityWidgets.get(entityWidgets.size() - 1);

            // Top section should always have mobs, prevent scrolling the entire list down the screen
            if ((scrollY >= 0 && firstPos >= 35) || (scrollY <= 0 && lastWidget.m_252907_() <= getWindow().m_85446_() - lastWidget.m_93694_())) {
                return false;
            }

            for (NarratableEntry button : ((ScreenAccessor) this).getSelectables()) {
                if (button instanceof EntityWidget<?> widget) {
                    widget.m_264152_(widget.m_252754_(), (int) (widget.m_252907_() + scrollY * 10));
                }
            }
        }

        return false;
    }

    @SuppressWarnings("unchecked")
    private synchronized void populateEntityWidgets(List<ShapeType<?>> rendered) {
        entityWidgets.clear();
        // add widget for each entity to be rendered
        int x = 15;
        int y = 35;
        int rows = (int) Math.ceil(rendered.size() / 7f);

        ShapeType<LivingEntity> currentType = ShapeType.from(PlayerShape.getCurrentShape(f_96541_.f_91074_));

        for (int yIndex = 0; yIndex <= rows; yIndex++) {
            for (int xIndex = 0; xIndex < 7; xIndex++) {
                int listIndex = yIndex * 7 + xIndex;

                if (listIndex < rendered.size()) {
                    ShapeType<?> type = rendered.get(listIndex);
                    Mob entity = renderEntities.get(type);
                    if (entity != null) {
                        EntityWidget<?> entityWidget = new EntityWidget<>(
                                (getWindow().m_85445_() - 27) / 7f * xIndex + x,
                                getWindow().m_85446_() / 5f * yIndex + y,
                                (getWindow().m_85445_() - 27) / 7f,
                                getWindow().m_85446_() / 5f,
                                (ShapeType<Mob>) type,
                                entity,
                                this,
                                ((RemorphedPlayerDataProvider) f_96541_.f_91074_).remorphed$getFavorites().contains(type),
                                type.equals(currentType)
                        );

                        m_142416_(entityWidget);
                        entityWidgets.add(entityWidget);
                    } else {
                        Remorphed.LOGGER.error("invalid shape type: " + type.getEntityType().m_20675_());
                    }
                }
            }
        }
    }

    public synchronized void populateUnlockedRenderEntities(Player player) {
        unlocked.clear();
        renderEntities.clear();
        List<ShapeType<?>> validUnlocked = Remorphed.getUnlockedShapes(player);
        for (ShapeType<?> type : validUnlocked) {
            Entity entity = type.create(Minecraft.m_91087_().f_91073_);
            if (entity instanceof Mob living) {
                unlocked.add(type);
                renderEntities.put(type, living);
            }
        }

        Remorphed.LOGGER.info(String.format("Loaded %d entities for rendering", unlocked.size()));
    }

    private SearchWidget createSearchBar() {
        return new SearchWidget(
                getWindow().m_85445_() / 2f - (getWindow().m_85445_() / 4f / 2) - 5,
                5,
                getWindow().m_85445_() / 4f,
                20f);
    }

    private Button createHelpButton() {
        Button.Builder helpButton = Button.m_253074_(Component.m_130674_("?"), (widget) -> Minecraft.m_91087_().m_91152_(new RemorphedHelpScreen()));

        int xOffset = Walkers.hasSpecialShape(Minecraft.m_91087_().f_91074_.m_20148_()) ? 30 : 0;

        helpButton.m_252794_((int) (getWindow().m_85445_() / 2f + (getWindow().m_85445_() / 8f) + 35 + xOffset), 5);
        helpButton.m_253046_(20, 20);
        helpButton.m_257505_(Tooltip.m_257550_(Component.m_237115_(Remorphed.MODID + ".help")));
        return helpButton.m_253136_();
    }

    private Button createVariantsButton() {
        Button.Builder variantsButton = Button.m_253074_(Component.m_237115_("remorphed.display_variants"), (widget) -> {
            Remorphed.displayVariantsInMenu = !Remorphed.displayVariantsInMenu;
            Minecraft.m_91087_().m_91152_(new RemorphedScreen());
        });

        variantsButton.m_252794_((int) (getWindow().m_85445_() / 2f - (getWindow().m_85445_() / 4f / 2) - 110), 5);
        variantsButton.m_253046_(100, 20);
        variantsButton.m_257505_(Tooltip.m_257550_(Component.m_237115_(Remorphed.MODID + ".variants")));

        return variantsButton.m_253136_();
    }

    private Button createSkillsButton() {
        Button.Builder skillButton = Button.m_253074_(Component.m_237115_("remorphed.show_skills"), (widget) -> Remorphed.displaySkillsInMenu = !Remorphed.displaySkillsInMenu);
        int xOffset = Walkers.hasSpecialShape(Minecraft.m_91087_().f_91074_.m_20148_()) ? 30 : 0;

        int xPos = (int) (getWindow().m_85445_() / 2f + (getWindow().m_85445_() / 8f) + 65 + xOffset);
        skillButton.m_252794_(xPos, 5);
        skillButton.m_253046_(Math.min(50, getWindow().m_85445_() - xPos), 20);
        skillButton.m_257505_(Tooltip.m_257550_(Component.m_237115_(Remorphed.MODID + ".skills")));

        return skillButton.m_253136_();
    }

    private PlayerWidget createPlayerButton() {
        return new PlayerWidget(
                (int) (getWindow().m_85445_() / 2f + (getWindow().m_85445_() / 8f) + 5),
                5,
                20,
                20,
                this);
    }

    private SpecialShapeWidget createSpecialShapeButton() {
        return new SpecialShapeWidget(
                (int) (getWindow().m_85445_() / 2f + (getWindow().m_85445_() / 8f) + 35),
                5,
                20,
                20,
                this);
    }

    private Window getWindow() {
        return Minecraft.m_91087_().m_91268_();
    }

    @Override
    public boolean m_6375_(double mouseX, double mouseY, int button) {
        if (mouseY < 35) {
            return searchBar.m_6375_(mouseX, mouseY, button) || helpButton.m_6375_(mouseX, mouseY, button) || variantsButton.m_6375_(mouseX, mouseY, button) || skillsButton.m_6375_(mouseX, mouseY, button) || playerButton.m_6375_(mouseX, mouseY, button) || specialShapeButton.m_6375_(mouseX, mouseY, button);
        } else {
            return super.m_6375_(mouseX, mouseY, button);
        }
    }
}
